<?php

namespace app\models;

use Yii;

/**
 * This is the model class for table "orders".
 *
 * @property int $id
 * @property int|null $user_id
 * @property int $service_id
 * @property string $address
 * @property string $payment_method
 * @property string|null $additional_info
 * @property string $status
 * @property string $created_at
 *
 * @property OrderDetail[] $orderDetails
 * @property Service $service
 * @property User $user
 */
class Order extends \yii\db\ActiveRecord
{
    // Константы статусов
    const STATUS_PENDING = 'pending';
    const STATUS_CONFIRMED = 'confirmed';
    const STATUS_IN_PROGRESS = 'in_progress';
    const STATUS_COMPLETED = 'completed';
    const STATUS_CANCELLED = 'cancelled';

    // Константы способов оплаты
    const PAYMENT_CASH = 'cash';
    const PAYMENT_CARD = 'card';

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'orders';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['user_id', 'service_id'], 'integer'],
            [['service_id', 'address', 'payment_method'], 'required'],
            [['additional_info'], 'string'],
            [['created_at'], 'safe'],
            [['address'], 'string', 'max' => 500],
            [['payment_method'], 'string', 'max' => 4],
            [['status'], 'string', 'max' => 20],
            [['status'], 'default', 'value' => self::STATUS_PENDING],
            [['payment_method'], 'in', 'range' => [self::PAYMENT_CASH, self::PAYMENT_CARD]],
            [['status'], 'in', 'range' => [
                self::STATUS_PENDING,
                self::STATUS_CONFIRMED,
                self::STATUS_IN_PROGRESS,
                self::STATUS_COMPLETED,
                self::STATUS_CANCELLED
            ]],
            [['service_id'], 'exist', 'skipOnError' => true,
                'targetClass' => Service::class, 'targetAttribute' => ['service_id' => 'id']],
            [['user_id'], 'exist', 'skipOnError' => true,
                'targetClass' => User::class, 'targetAttribute' => ['user_id' => 'id']],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'user_id' => 'Пользователь',
            'service_id' => 'Услуга',
            'address' => 'Адрес',
            'payment_method' => 'Способ оплаты',
            'additional_info' => 'Дополнительная информация',
            'status' => 'Статус',
            'created_at' => 'Дата создания',
        ];
    }

    /**
     * Gets query for [[OrderDetails]].
     *
     * @return \yii\db\ActiveQuery
     */
    public function getOrderDetails()
    {
        return $this->hasMany(OrderDetail::class, ['order_id' => 'id']);
    }

    /**
     * Gets query for [[Service]].
     *
     * @return \yii\db\ActiveQuery
     */
    public function getService()
    {
        return $this->hasOne(Service::class, ['id' => 'service_id']);
    }

    /**
     * Gets query for [[User]].
     *
     * @return \yii\db\ActiveQuery
     */
    public function getUser()
    {
        return $this->hasOne(User::class, ['id' => 'user_id']);
    }

    /**
     * Получает текст статуса
     * @return string
     */
    public function getStatusText()
    {
        $statuses = self::getStatusList();
        return $statuses[$this->status] ?? $this->status;
    }

    /**
     * Получает текст способа оплаты
     * @return string
     */
    public function getPaymentMethodText()
    {
        $methods = self::getPaymentMethods();
        return $methods[$this->payment_method] ?? $this->payment_method;
    }

    /**
     * Получает список статусов
     * @return array
     */
    public static function getStatusList()
    {
        return [
            self::STATUS_PENDING => 'Новая',
            self::STATUS_CONFIRMED => 'Подтверждена',
            self::STATUS_IN_PROGRESS => 'В работе',
            self::STATUS_COMPLETED => 'Завершена',
            self::STATUS_CANCELLED => 'Отменена',
        ];
    }

    /**
     * Получает список способов оплаты
     * @return array
     */
    public static function getPaymentMethods()
    {
        return [
            self::PAYMENT_CASH => 'Наличный расчет',
            self::PAYMENT_CARD => 'Безналичный расчет',
        ];
    }

    /**
     * Получает детали заказа сгруппированные по названиям атрибутов
     * @return array
     */
    public function getDetailsWithAttributes()
    {
        $details = [];

        foreach ($this->orderDetails as $detail) {
            if ($detail->serviceAttribute) {
                $details[$detail->serviceAttribute->attribute_name] = $detail->value;
            }
        }

        return $details;
    }

    /**
     * Получает имя пользователя или "Гость"
     * @return string
     */
    public function getUserName()
    {
        if ($this->user) {
            return $this->user->getFullName();
        }
        return 'Гость';
    }

    /**
     * {@inheritdoc}
     */
    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($insert) {
                $this->created_at = date('Y-m-d H:i:s');
            }
            return true;
        }
        return false;
    }
}