<?php

namespace app\controllers;

use Yii;
use yii\bootstrap5\Html;
use yii\web\Controller;
use yii\web\Response;
use yii\widgets\ActiveForm;
use app\models\OrderForm;
use app\models\Order;
use app\models\OrderDetail;
use app\models\Service;
use app\models\ServiceAttribute;
use app\models\AttributeOption;
use app\models\User;
use yii\helpers\ArrayHelper;
use yii\web\NotFoundHttpException;

class OrderController extends Controller
{
    /**
     * Список заявок пользователя
     */
    public function actionIndex()
    {
        if (Yii::$app->user->isGuest) {
            return $this->redirect(['site/login']);
        }

        $orders = Order::find()
            ->where(['user_id' => Yii::$app->user->id])
            ->with('service') // Получаем связанную услугу
            ->orderBy(['created_at' => SORT_DESC])
            ->all();

        return $this->render('index', [
            'orders' => $orders,
        ]);
    }

    /**
     * Страница создания заявки
     */
    public function actionCreate()
    {
        $model = new OrderForm();

        // Если пользователь авторизован, заполняем user_id
        if (!Yii::$app->user->isGuest) {
            $user = User::findOne(Yii::$app->user->id);
            if ($user) {
                $model->user_id = $user->id;
            }
        }

        // Обработка AJAX-валидации
        if (Yii::$app->request->isAjax && $model->load(Yii::$app->request->post())) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }

        // Обработка отправки формы
        if ($model->load(Yii::$app->request->post())) {
            // Проверяем чекбокс дополнительной информации
            $hasAdditionalInfo = Yii::$app->request->post('OrderForm')['has_additional_info'] ?? 0;
            $model->has_additional_info = $hasAdditionalInfo;

            if ($model->validate()) {
                // Создаем заявку в базе данных
                $transaction = Yii::$app->db->beginTransaction();

                try {
                    // Сохраняем основную заявку
                    $order = new Order();
                    $order->user_id = $model->user_id ?: null;
                    $order->service_id = $model->service_id;
                    $order->address = $model->address;
                    $order->payment_method = $model->payment_method;
                    $order->additional_info = $model->has_additional_info ? $model->additional_info : null;
                    $order->status = Order::STATUS_PENDING;

                    if ($order->save()) {
                        // Сохраняем динамические атрибуты
                        $attributes = Yii::$app->request->post('OrderForm')['attributes'] ?? [];

                        foreach ($attributes as $attributeId => $value) {
                            // Пропускаем пустые значения (кроме чекбоксов, они могут быть 0)
                            if (empty($value) && $value !== '0') {
                                continue;
                            }

                            // Проверяем существование атрибута
                            $attribute = ServiceAttribute::findOne($attributeId);
                            if (!$attribute) {
                                continue;
                            }

                            // Для чекбокса проверяем значение
                            if ($attribute->attribute_type === 'checkbox' && empty($value)) {
                                continue;
                            }

                            // Для select получаем текстовое значение опции
                            if ($attribute->attribute_type === 'select' && !empty($value)) {
                                $option = AttributeOption::findOne($value);
                                if ($option) {
                                    $value = $option->option_value;
                                }
                            }

                            $orderDetail = new OrderDetail();
                            $orderDetail->order_id = $order->id;
                            $orderDetail->attribute_id = $attributeId;
                            $orderDetail->value = (string)$value;

                            if (!$orderDetail->save()) {
                                Yii::error('Ошибка сохранения детали заказа: ' . print_r($orderDetail->errors, true), 'order');
                            }
                        }

                        $transaction->commit();

                        Yii::$app->session->setFlash('success', 'Ваша заявка успешно создана! Номер заявки: ' . $order->id);

                        // Редирект на страницу успеха или на список заявок
                        if (Yii::$app->user->isGuest) {
                            return $this->redirect(['site/index']);
                        } else {
                            return $this->redirect(['view', 'id' => $order->id]);
                        }
                    } else {
                        throw new \Exception('Ошибка сохранения заявки: ' . print_r($order->errors, true));
                    }

                } catch (\Exception $e) {
                    $transaction->rollBack();
                    Yii::error('Ошибка создания заявки: ' . $e->getMessage(), 'order');
                    Yii::$app->session->setFlash('error', 'Произошла ошибка при создании заявки. Пожалуйста, попробуйте еще раз.');
                }
            } else {
                Yii::$app->session->setFlash('error', 'Пожалуйста, исправьте ошибки в форме.');
            }
        }

        // Получаем список услуг для dropdown
        $services = Service::find()->all();
        $serviceList = ArrayHelper::map($services, 'id', 'name');

        return $this->render('create', [
            'model' => $model,
            'serviceList' => $serviceList,
        ]);
    }

    /**
     * AJAX-запрос для получения динамических полей
     */
    public function actionGetAttributes($service_id)
    {
        Yii::$app->response->format = Response::FORMAT_HTML;

        try {
            // Находим услугу
            $service = Service::findOne($service_id);
            if (!$service) {
                return '<div class="alert alert-danger">Услуга не найдена</div>';
            }

            // Получаем атрибуты для этой услуги
            $attributes = ServiceAttribute::find()
                ->where(['service_id' => $service_id])
                ->with('attributeOptions')
                ->orderBy(['id' => SORT_ASC])
                ->all();

            if (empty($attributes)) {
                return '<div class="alert alert-info">Нет дополнительных полей для "' . Html::encode($service->name) . '"</div>';
            }

            $html = '<div class="category-fields">';
            $html .= '<h4>Детали услуги "' . Html::encode($service->name) . '":</h4>';

            foreach ($attributes as $attribute) {
                $fieldName = 'OrderForm[attributes][' . $attribute->id . ']';
                $fieldId = 'attribute-' . $attribute->id;

                $html .= '<div class="form-group">';
                $html .= '<label class="control-label">' . Html::encode($attribute->attribute_name) . '</label>';

                switch ($attribute->attribute_type) {
                    case 'text':
                        $html .= '<input type="text" class="form-control" name="' . $fieldName . '" id="' . $fieldId . '">';
                        break;

                    case 'select':
                        $html .= '<select class="form-control" name="' . $fieldName . '" id="' . $fieldId . '">';
                        $html .= '<option value="">Выберите...</option>';

                        if (!empty($attribute->attributeOptions)) {
                            foreach ($attribute->attributeOptions as $option) {
                                $html .= '<option value="' . Html::encode($option->id) . '">';
                                $html .= Html::encode($option->option_value);
                                $html .= '</option>';
                            }
                        }

                        $html .= '</select>';
                        break;

                    case 'number':
                        $html .= '<input type="number" class="form-control" name="' . $fieldName . '" id="' . $fieldId . '" min="0">';
                        break;

                    case 'checkbox':
                        $html .= '<div class="checkbox">';
                        $html .= '<label>';
                        $html .= '<input type="checkbox" name="' . $fieldName . '" value="1"> ';
                        $html .= Html::encode($attribute->attribute_name);
                        $html .= '</label>';
                        $html .= '</div>';
                        break;
                }

                $html .= '</div>';
            }

            $html .= '</div>';
            return $html;

        } catch (\Exception $e) {
            Yii::error('Ошибка в actionGetAttributes: ' . $e->getMessage(), 'order');
            return '<div class="alert alert-danger">Ошибка загрузки полей: ' . Html::encode($e->getMessage()) . '</div>';
        }
    }

    /**
     * Дефолтные опции для атрибутов, если их нет в БД
     */
    private function getDefaultOptions($attribute)
    {
        $defaults = [
            'Вид одежды' => ['Пальто', 'Куртка', 'Плащ', 'Пуховик', 'Шуба'],
            'Материал одежды' => ['Шерсть', 'Хлопок', 'Синтетика', 'Кожа', 'Замша'],
            'Вид обуви' => ['Кроссовки', 'Туфли', 'Сапоги', 'Ботинки'],
            'Материал обуви' => ['Кожа', 'Замша', 'Текстиль', 'Резина'],
            'Вид мебели' => ['Диван', 'Кресло', 'Кровать', 'Стул'],
            'Материал мебели' => ['Ткань', 'Кожа', 'Искусственная кожа'],
            'Степень загрязнения' => ['Легкое', 'Среднее', 'Сильное'],
            'Материал ковра' => ['Шерсть', 'Синтетика', 'Хлопок', 'Смешанный'],
        ];

        $attributeName = $attribute->attribute_name;
        $html = '';

        if (isset($defaults[$attributeName])) {
            foreach ($defaults[$attributeName] as $index => $option) {
                $html .= '<option value="' . ($index + 1) . '">' . Html::encode($option) . '</option>';
            }
        }

        return $html;
    }

    /**
     * AJAX-валидация формы
     */
    public function actionValidate()
    {
        $model = new OrderForm();
        if (Yii::$app->request->isAjax && $model->load(Yii::$app->request->post())) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }
        return '';
    }

    /**
     * Просмотр заявки (для авторизованных пользователей)
     */
    public function actionView($id)
    {
        if (Yii::$app->user->isGuest) {
            return $this->redirect(['site/login']);
        }

        $order = Order::find()
            ->where(['id' => $id])
            ->andWhere(['user_id' => Yii::$app->user->id])
            ->with('service') // Получаем связанную услугу
            ->one();

        if (!$order) {
            throw new NotFoundHttpException('Заявка не найдена или у вас нет доступа');
        }

        // Получаем детали заявки БЕЗ связи с serviceAttribute
        $orderDetails = OrderDetail::find()
            ->where(['order_id' => $id])
            ->all(); // ← Убрали with()

        // Если нужны названия атрибутов, получаем их отдельно
        $attributeIds = [];
        foreach ($orderDetails as $detail) {
            $attributeIds[] = $detail->attribute_id;
        }

        $attributes = [];
        if (!empty($attributeIds)) {
            $attributes = ServiceAttribute::find()
                ->where(['id' => $attributeIds])
                ->indexBy('id') // Индексируем по ID для быстрого поиска
                ->all();
        }

        return $this->render('view', [
            'order' => $order,
            'orderDetails' => $orderDetails,
            'attributes' => $attributes, // Передаем атрибуты отдельно
        ]);
    }

    /**
     * Страница успешного создания заявки (для гостей)
     */
    public function actionSuccess($id)
    {
        $order = Order::findOne($id);

        if (!$order) {
            throw new NotFoundHttpException('Заявка не найдена');
        }

        // Получаем детали заявки БЕЗ связи с serviceAttribute
        $orderDetails = OrderDetail::find()
            ->where(['order_id' => $id])
            ->all(); // ← Убрали with()

        // Если нужны названия атрибутов, получаем их отдельно
        $attributeIds = [];
        foreach ($orderDetails as $detail) {
            $attributeIds[] = $detail->attribute_id;
        }

        $attributes = [];
        if (!empty($attributeIds)) {
            $attributes = ServiceAttribute::find()
                ->where(['id' => $attributeIds])
                ->indexBy('id')
                ->all();
        }

        return $this->render('success', [
            'order' => $order,
            'orderDetails' => $orderDetails,
            'attributes' => $attributes,
        ]);
    }
}